import { NextRequest, NextResponse } from 'next/server';
import Database from 'better-sqlite3';
import path from 'path';

const dbPath = path.join(process.cwd(), 'domains.db');

// PUT /api/project-tags/[id] - עדכון תגית
export async function PUT(
    request: NextRequest,
    { params }: { params: { id: string } }
) {
    try {
        const db = new Database(dbPath);
        const { id } = params;
        const body = await request.json();

        const { name, color, description } = body;

        // בדיקה שהתגית קיימת
        const existingTag = db.prepare('SELECT id FROM project_tags WHERE id = ?').get(id);
        if (!existingTag) {
            db.close();
            return NextResponse.json(
                { error: 'תגית לא נמצאה' },
                { status: 404 }
            );
        }

        const updateQuery = `
      UPDATE project_tags SET
        name = COALESCE(?, name),
        color = COALESCE(?, color),
        description = COALESCE(?, description)
      WHERE id = ?
    `;

        db.prepare(updateQuery).run(name, color, description, id);

        db.close();

        return NextResponse.json({
            message: 'תגית עודכנה בהצלחה'
        });

    } catch (error) {
        console.error('Error updating project tag:', error);
        return NextResponse.json(
            { error: 'שגיאה בעדכון התגית' },
            { status: 500 }
        );
    }
}

// DELETE /api/project-tags/[id] - מחיקת תגית
export async function DELETE(
    request: NextRequest,
    { params }: { params: { id: string } }
) {
    try {
        const db = new Database(dbPath);
        const { id } = params;

        // בדיקה שהתגית קיימת
        const existingTag = db.prepare('SELECT id FROM project_tags WHERE id = ?').get(id);
        if (!existingTag) {
            db.close();
            return NextResponse.json(
                { error: 'תגית לא נמצאה' },
                { status: 404 }
            );
        }

        db.prepare('DELETE FROM project_tags WHERE id = ?').run(id);

        db.close();

        return NextResponse.json({
            message: 'תגית נמחקה בהצלחה'
        });

    } catch (error) {
        console.error('Error deleting project tag:', error);
        return NextResponse.json(
            { error: 'שגיאה במחיקת התגית' },
            { status: 500 }
        );
    }
}
